#include <iostream>
#include <vector>
#include <utility>

#define endl '\n'

using namespace std;

struct Edge
{
    int to, dist;

    Edge()
    {
        to = -1;
        dist = 0;
    }

    Edge(int _to, int _dist)
    {
        to = _to;
        dist = _dist;
    }
};

const int MAXN = 2e5 + 3;
const int MAX_LOG = 20;

vector<Edge> graph[MAXN];
int original_pos[MAXN];

Edge parent[MAXN];
bool visited[MAXN];

vector<Edge> cycle;
int nearest_cycle_index[MAXN];
int max_dist_to_cycle[MAXN];

int rmq[MAXN][MAX_LOG];
int log_values[MAXN];

bool find_cycle(int currv)
{
    visited[currv] = true;

    for (Edge& nextv : graph[currv])
    {
        if (nextv.to == parent[currv].to)
        {
            continue;
        }

        if (visited[nextv.to])
        {
            cycle.push_back(nextv);

            while (currv != nextv.to)
            {
                cycle.push_back(Edge(currv, parent[currv].dist));
                currv = parent[currv].to;
            }

            return true;
        }

        parent[nextv.to] = Edge(currv, nextv.dist);
        if (find_cycle(nextv.to))
        {
            return true;
        }
    }

    return false;
}

bool find_nearest_cycle_index(int currv, int curr_parent)
{
    if (nearest_cycle_index[currv] != -1)
    {
        return true;
    }

    for (Edge& nextv : graph[currv])
    {
        if (nextv.to == curr_parent)
        {
            continue;
        }

        if (find_nearest_cycle_index(nextv.to, currv))
        {
            nearest_cycle_index[currv] = nearest_cycle_index[nextv.to];
            max_dist_to_cycle[currv] = max(nextv.dist, max_dist_to_cycle[nextv.to]);
        }
    }

    return false;
}

void build_rmq(int cycle_size)
{
    log_values[0] = log_values[1] = 0;
    for (int i = 2; i <= cycle_size; ++i)
    {
        log_values[i] = log_values[i / 2] + 1;
    }

    for (int i = 0; i < cycle_size; ++i)
    {
        rmq[i][0] = cycle[i].dist;
    }

    for (int range = 1; range < MAX_LOG; ++range)
    {
        for (int i = 0; i < cycle_size; ++i)
        {
            rmq[i][range] = rmq[i][range - 1];

            int j = i + (1 << (range - 1));
            if (j < cycle_size)
            {
                rmq[i][range] = max(rmq[i][range], rmq[j][range - 1]);
            }
        }
    }
}

int rmq_query(int l, int r)
{
    if (l > r)
    {
        return 0;
    }

    int range = log_values[r - l + 1];
    return max(rmq[l][range], rmq[r - (1 << range) + 1][range]);
}

int64_t answer_query(int x, int y, int cycle_size)
{
    int outside_cycle = max(max_dist_to_cycle[x], max_dist_to_cycle[y]);

    x = nearest_cycle_index[x];
    y = nearest_cycle_index[y];

    if (x > y) {
        swap(x, y);
    }

    int inside_cycle1 = rmq_query(x, y - 1);
    int inside_cycle2 = max(rmq_query(0, x - 1), rmq_query(y, cycle_size - 1));

    int bigger_dist = max(outside_cycle, max(inside_cycle1, inside_cycle2));
    int smaller_dist = max(outside_cycle, min(inside_cycle1, inside_cycle2));

    return 2ll * bigger_dist + 1ll * smaller_dist;
}

int main()
{
    ios::sync_with_stdio(false);
    cin.tie(nullptr);

    int n;
    cin >> n;

    for (int i = 0; i < n; ++i)
    {
        int from, to, dist;
        cin >> from >> to >> dist;

        --from;
        --to;

        graph[from].push_back(Edge(to, dist));
        graph[to].push_back(Edge(from, dist));
    }

    for (int i = 0; i < n; ++i)
    {
        original_pos[i] = i;
        nearest_cycle_index[i] = -1;
    }

    parent[0] = Edge(-1, 0);
    find_cycle(0);

    int cycle_size = (int) cycle.size();
    for (int i = 0; i < cycle_size; ++i)
    {
        nearest_cycle_index[cycle[i].to] = i;
        max_dist_to_cycle[cycle[i].to] = 0;
    }

    for (int i = 0; i < n; ++i)
    {
        if (nearest_cycle_index[i] == -1)
        {
            find_nearest_cycle_index(i, -1);
        }
    }

    build_rmq(cycle_size);

    int q;
    cin >> q;

    int64_t res = 0;
    for (int i = 0; i < q; ++i)
    {
        int type, x, y;
        cin >> type >> x >> y;

        --x;
        --y;

        if (type == 2)
        {
            swap(original_pos[x], original_pos[y]);
            continue;
        }

        x = original_pos[x];
        y = original_pos[y];

        res += answer_query(x, y, cycle_size);
    }

    cout << res << endl;
}
